//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
//  Sample MAP ARX API application. (Thorough error checking is ommited.)
//
#include <afx.h>
#include <stdio.h>
#include <MapArxApi.h>
#include <MapSession.h>
#include <MapProj.h>
#include <MapReactors.h>
#include <MapBoundary.h>
#include <MapQuery.h>
#include <MapAlteration.h>
#include <MapValue.h>
#include <MapTemplate.h>
#include <MapStringArray.h>
#include <gemat3d.h>
#include <accmd.h>
#include <adscodes.h>
#include <rxregsvc.h>        

// Main ARX entry point
extern "C" {
   AcRx::AppRetCode acrxEntryPoint(AcRx::AppMsgCode msg, void* ptr);
}

///////////////////////////////////////////////////////////
/*					Map Reactors                         */
///////////////////////////////////////////////////////////

// Custom Drawing Set Reactor class
class AcMyDrawingSetReactor : public AcMapDrawingSetReactor
{
public:
						AcMyDrawingSetReactor() ;

	virtual				~AcMyDrawingSetReactor() ;

    // The notification that the drawing is attached to the Drawing Set
    virtual void        DrawingAttached     (const AcMapAttachedDrawing *) ;
    
    // The notification that the drawing is detached from the Drawing Set
    virtual void        DrawingDetached    (unsigned long Id, const char *pcName, int level) ;
    
    // The notification that the  drawing is activated
    virtual void        DrawingActivated    (const AcMapAttachedDrawing *) ;
    
    // The notification that the drawing is deactivated
    virtual void        DrawingDeactivated  (const AcMapAttachedDrawing *) ;

    // The notification that the drawing settings are modified
    virtual void        DrawingSettingsModified 
											(const AcMapAttachedDrawing *) ;

} ;

// Custom Alias List Reactor class
class AcMyAliasesReactor : public AcMapAliasesReactor
{
public:
						AcMyAliasesReactor() ;

	virtual				~AcMyAliasesReactor() ;

    // The notification that the drawing is attached to the Drawing Set
    virtual void        AliasAdded(const AcMapDriveAlias *pObj) ;
    
    // The notification that the drawing is detached from the Drawing Set
    virtual void        AliasDeleted(const char *, const char *) ;
} ;

// Custom Query Library Reactor class
class AcMyQueryLibraryReactor : public AcMapQueryLibraryReactor
{
public:
						AcMyQueryLibraryReactor() ;

	virtual				~AcMyQueryLibraryReactor() ;

    // The notification that a query category is added.
    virtual void        QueryCategoryAdded	(const AcMapQueryCategory *pObj) ;
    
    // The notification that a query category is deleted.
    virtual void        QueryCategoryDeleted(unsigned long Id, 
											 const char *pcName) ;
    
    // The notification that a query category is renamed.
    virtual void        QueryCategoryRenamed(const AcMapQueryCategory *pObj) ;
    
    // The notification that a query is added.
    virtual void        QueryAdded		    (const AcMapQueryAttribute *pObj) ;
                                             
    // The notification that a query is deleted.
    virtual void        QueryDeleted	    (unsigned long QueryId,
											 const char *pcName) ;

    virtual void        QueryRenamed		(const AcMapQueryAttribute *pObj) ;

    virtual void        QueryModified		(const AcMapQueryAttribute *pObj) ;
} ;

// Custom Project Options Reactor class
class AcMyProjectOptionsReactor : public AcMapProjectOptionsReactor
{
public:
						AcMyProjectOptionsReactor() ;

	virtual				~AcMyProjectOptionsReactor() ;

    // The notification that the drawing is attached to the Drawing Set
    virtual void        OptionModified(const char *pcName, const char *pcOldVal, const char *pcNewVal) ;
	virtual void        OptionModified(const char *pcName, int iOldVal, int iNewVal) ;
};

// Custom Session Options Reactor class
class AcMySessionOptionsReactor : public AcMapSessionOptionsReactor
{
public:
						AcMySessionOptionsReactor() ;

	virtual				~AcMySessionOptionsReactor() ;

    // The notification that the drawing is attached to the Drawing Set
    virtual void        OptionModified(const char *pcName, const char *pcOldVal, const char *pcNewVal) ;
	virtual void        OptionModified(const char *pcName, int iOldVal, int iNewVal) ;
};

// Custom Drawing Set Reactor class implementation
AcMyDrawingSetReactor::AcMyDrawingSetReactor() : AcMapDrawingSetReactor()
{}

AcMyDrawingSetReactor::~AcMyDrawingSetReactor()
{}

void        
AcMyDrawingSetReactor::DrawingAttached(
	const AcMapAttachedDrawing *pObj
)
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing attached %s", pcName) ;
	}
}
    
void        
AcMyDrawingSetReactor::DrawingDetached(
	unsigned long Id,
	const char *pcName,
	int level
)
{
	acutPrintf ("\n***NOTIFICATION:Drawing %s detached", pcName) ;
}
    
void        
AcMyDrawingSetReactor::DrawingActivated(
	const AcMapAttachedDrawing *pObj
)
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing activated %s", pcName) ;
	}
}
    
void        
AcMyDrawingSetReactor::DrawingDeactivated(
	const AcMapAttachedDrawing *pObj
)
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing deactivated %s", pcName) ;
	}
}

void        
AcMyDrawingSetReactor::DrawingSettingsModified( 
	const AcMapAttachedDrawing *pObj
) 
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing settings modified %s", pcName) ;
	}
}

// Custom Alias List Reactor class implementation
AcMyAliasesReactor::AcMyAliasesReactor() : AcMapAliasesReactor()
{}

AcMyAliasesReactor::~AcMyAliasesReactor()
{}

void        
AcMyAliasesReactor::AliasAdded(
	const AcMapDriveAlias *pObj
)
{
	const char *pcName ;
	pObj->GetName(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Alias added %s", pcName) ;
	}
}
    
void        
AcMyAliasesReactor::AliasDeleted(
	const char *pcName, 
	const char *pcPath
)
{
	acutPrintf ("\nNOTIFICATION: Alias %s \t= %s is deleted.", pcName, pcPath) ;
}

// Custom Query Library Reactor class implementation
AcMyQueryLibraryReactor::AcMyQueryLibraryReactor() 
{}

AcMyQueryLibraryReactor::~AcMyQueryLibraryReactor() 
{}

void        
AcMyQueryLibraryReactor::QueryCategoryAdded	(
	const AcMapQueryCategory *pObj
)
{
	const char *pcName ;
	if (pObj->GetName(pcName) == AcMap::kOk)
	{
		acutPrintf("\n***NOTIFICATION: Query category %s was added.", pcName) ;
	}
}

void        
AcMyQueryLibraryReactor::QueryCategoryDeleted(unsigned long Id, const char *pcName)
{
	if (pcName)
	{
		acutPrintf("\n***NOTIFICATION: Query category %s was deleted.", pcName) ;
	}
}

void        
AcMyQueryLibraryReactor::QueryCategoryRenamed(
	const AcMapQueryCategory *pObj
)
{
	const char *pcName ;
	if (pObj->GetName(pcName) == AcMap::kOk)
	{
		acutPrintf("\n***NOTIFICATION: Query category %s was renamed.", pcName) ;
	}
}

void        
AcMyQueryLibraryReactor::QueryAdded(
	const AcMapQueryAttribute *pObj
)
{
	const char *pcName ;
	if (pObj->GetName(pcName) == AcMap::kOk)
	{
		acutPrintf("\n***NOTIFICATION: Query %s was added.", pcName) ;
	}
}

void        
AcMyQueryLibraryReactor::QueryDeleted(
	unsigned long QueryId,
	const char *pcName
)
{
	if (pcName)
	{
		acutPrintf("\n***NOTIFICATION: Query %s was deleted.", pcName) ;
	}
}

void        
AcMyQueryLibraryReactor::QueryRenamed(
	const AcMapQueryAttribute *pObj
)
{
	const char *pcName ;
	if (pObj->GetName(pcName) == AcMap::kOk)
	{
		acutPrintf("\n***NOTIFICATION: Query %s was renamed.", pcName) ;
	}
}

void        
AcMyQueryLibraryReactor::QueryModified(
	const AcMapQueryAttribute *pObj
)
{
	const char *pcName ;
	if (pObj->GetName(pcName) == AcMap::kOk)
	{
		acutPrintf("\n***NOTIFICATION: Query %s was modified.", pcName) ;
	}
}

// Custom Project Options Reactor class implementation
AcMyProjectOptionsReactor::AcMyProjectOptionsReactor() 
{}

AcMyProjectOptionsReactor::~AcMyProjectOptionsReactor()
{}

void        
AcMyProjectOptionsReactor::OptionModified(
	const char *pcName, 
	const char *pcOldVal, 
	const char *pcNewVal
) 
{
	acutPrintf("\nOption %s old: %s new: %s", pcName, pcOldVal, pcNewVal) ;
}

void        
AcMyProjectOptionsReactor::OptionModified(
	const char *pcName, 
	int iOldVal, 
	int iNewVal
) 
{
	acutPrintf("\nOption %s old: %d new: %d", pcName, iOldVal, iNewVal) ;
}

// Custom Session Options Reactor class implementation
AcMySessionOptionsReactor::AcMySessionOptionsReactor()
{}

AcMySessionOptionsReactor::~AcMySessionOptionsReactor()
{}

void        
AcMySessionOptionsReactor::OptionModified(
	const char *pcName, 
	const char *pcOldVal, 
	const char *pcNewVal
) 
{
	acutPrintf("\nOption %s old: %s new: %s", pcName, pcOldVal, pcNewVal) ;
}

void        
AcMySessionOptionsReactor::OptionModified(
	const char *pcName, 
	int iOldVal, 
	int iNewVal
) 
{
	acutPrintf("\nOption %s old: %d new: %d", pcName, iOldVal, iNewVal) ;
}

///////////////////////////////////////////////////////////
/*					End Map Reactors                     */
///////////////////////////////////////////////////////////

// reactors initialization
static AcMyAliasesReactor *pAliasReact = NULL;
static AcMyDrawingSetReactor *pDSetReact = NULL;
static AcMyQueryLibraryReactor *pQLibReact = NULL;
static AcMyProjectOptionsReactor *pPrefReact = NULL;
static AcMySessionOptionsReactor  *pSysOptReactor = NULL;

// print MAP error stack
void printErrStack()
{
    AcMapSession    *mapApi ;
    AcMapErrorStack *pErr ;
	   
    do {
		// get the root API object
        mapApi = AcMapGetSession();
        if (mapApi == NULL) 
        {
			acutPrintf ("\nCan't connect to MAP") ;
			break ;
        }
        
		// get MAP error stack
		if (mapApi->GetErrorStack(pErr) == Adesk::kFalse)
        {
            acutPrintf ("\nCan't get error stack") ;
            break ;
        }
            
		if (pErr->CountEntries())
		{
			acutPrintf ("\nMAP ERRORS:\n******************************************") ;
		}

		// AcMapErrorEntry pointer should be initialized to NULL before the loop
		AcMapErrorEntry *pEntry = NULL ;
        for (int i = 0; i < pErr->CountEntries(); i++)
		{
			//get the error entry object
			if (pErr->GetEntry(pEntry, i))
			{
				//get the error message
				char *msg = NULL ;
				pEntry->GetErrorMessage(msg) ;
				acutPrintf("\nError %d: %s", pEntry->ErrorCode(), msg) ;
				delete [] msg;

				// AcMapErrorParameter pointer should be initialized to NULL before the loop
				AcMapErrorParameter *pPar = NULL ;
				for (int k = 0; k < pEntry->CountParameters(); k++)
				{
					//get the error parameter for the error entry
        			if (pEntry->GetParameter(pPar, k))
					{
						//get the specific error information
						char *pP = NULL,
							 *pSQL = NULL ;
						pPar->GetParameterDescription(pP) ;
						pPar->GetStatement(pSQL) ;
						if (pSQL && pP)
						{
							acutPrintf("\n\tParameter: %s = %s pos:%d", pP, pSQL, pPar->GetErrorPosition()) ;
						}
						else if(pP)
						{
							acutPrintf("\n\tParameter: %s", pP) ;
						}
						
						delete [] pP ;
						delete [] pSQL ;
					}

				}
				// this object was used within the loop and should be deleted only after the loop
				if(pPar)
					delete pPar ;
			}
        }

		// this object was used within the loop and should be deleted only after the loop
		delete pEntry ;
		
		pErr->Clear() ;
        
    } while (0) ;
}

// print alias list
void printAliases(AcMapAliases *pAliases)
{
	// AcMapDriveAlias pointer should be initialized to NULL before the loop
    AcMapDriveAlias *pAlias = NULL ;
    
    for (int i = 0; i < pAliases->CountAliases(); i++)
    {
        if (i == 0)
        {
            acutPrintf ("\n\tAliases:") ;
        }
        
		// be sure pAlias was initialized to NULL before the loop to be allocated at the 
		// first call
        if (pAliases->GetAlias(pAlias, i) == AcMap::kOk)
        {
            const char *name, *path ;
            pAlias->GetName(name) ;
            pAlias->GetPath(path) ;
            acutPrintf ("\n%s \t= %s", name, path) ;
        }
    }

	// this object was used within the loop and should be deleted only after the loop
	if(pAlias)
		delete pAlias ;
}

// print all atached drawings from the drawing set
void printDSet(AcMapDrawingSet *pDSet)
{
	// AcMapAttachedDrawing pointer should be initialized to NULL before the loop
    AcMapAttachedDrawing *pDEntry = NULL ;
    
    for (int i = 0; i < pDSet->CountDrawings(Adesk::kTrue); i++)
    {
        if (i == 0)
        {
            acutPrintf ("\n\tDrawing Set:") ;
        }

      	// be sure pDEntry was initialized to NULL before the loop to be allocated at the 
		// first call. Passing Adesk::kTrue as the 3-rd parameter makes this loop iterate only 
		// 0-level drawings.
        if (pDSet->GetDrawing(pDEntry, i, Adesk::kTrue) == AcMap::kOk)
        {
            const char *path , *desc;
            pDEntry->GetAliasPath(path) ;
            pDEntry->GetDescription(desc) ;
			switch (pDEntry->ActiveStatus())
			{
				case AcMap::kDwgInactive:
				acutPrintf ("\nNon %s (%s)", path, desc) ;
				break ;

				case AcMap::kDwgActive:
				acutPrintf ("\nAct %s (%s)", path, desc) ;
				break ;

				case AcMap::kDwgLocked:	
				acutPrintf ("\nLoc %s (%s)", path, desc) ;
				break ;
			}

			//check to see if we have nested drawings
			if (pDEntry->CountNestedDrawings(Adesk::kFalse))
			{
				// AcMapAttachedDrawing pointer should be initialized to NULL before the loop
				AcMapAttachedDrawing *pDNestEntry = NULL ;

				// iterate all nested drawings
				for (int i = 0; 
					 pDEntry->GetNestedDrawing(pDNestEntry, i, Adesk::kFalse) == AcMap::kOk;
					 i++)
				{
					int l = pDNestEntry->NestedLevel() ;
					acutPrintf ("\n") ;
					while(l--)
					{
						acutPrintf (" ") ;
					}
					pDNestEntry->GetAliasPath(path) ;
					pDNestEntry->GetDescription(desc) ;
					switch (pDNestEntry->ActiveStatus())
					{
						case AcMap::kDwgInactive:
						acutPrintf ("Non %s (%s)", path, desc) ;
						break ;

						case AcMap::kDwgActive:
						acutPrintf ("Act %s (%s)", path, desc) ;
						break ;

						case AcMap::kDwgLocked:	
						acutPrintf ("Loc %s (%s)", path, desc) ;
						break ;
					}
				}
				// this object was used within the loop and should be deleted only after the loop
				if(pDNestEntry)
					 delete pDNestEntry ;
			}
        }
	}
	// this object was used within the loop and should be deleted only after the loop
	if(pDEntry)
		delete pDEntry ;
}

// get the specified symbol table from the attached drawing
void
getTable(AcMapAttachedDrawing *pDwg)
{
    char				res[32] ;   

	while(acedInitGet (0, "eXit Layer Block lineType Regapp Odd Ase textStyle") == RTNORM
	      && acedGetKword("\nLayer/Block/lineType/Regapp/Odd/Ase/textStyle/<eXit>: ", res) == RTNORM)
   	{
		AcMapStringArray	tAr ;
		if (!(strcmp(res, "eXit")))
        {
			break ;
        }
        else if (!strcmp(res, "Layer"))
		{
			// get the list of layers defined in the source drawing
			if (pDwg->GetTableList(tAr, AcMap::kLayerTable) == AcMap::kOk)
			{
				acutPrintf("\nLayers:") ;
			}
		}
        else if (!strcmp(res, "Block"))
		{
			// get the list of blocks defined in the source drawing
			if (pDwg->GetTableList(tAr, AcMap::kBlockTable) == AcMap::kOk)
			{
				acutPrintf("\nBocks:") ;
			}
		}
        else if (!strcmp(res, "lineType"))
		{
			// get the list of linetypes defined in the source drawing
			if (pDwg->GetTableList(tAr, AcMap::kLinetypeTable) == AcMap::kOk)
			{
				acutPrintf("\nLinetypes:") ;
			}
		}
        else if (!strcmp(res, "Regapp"))
		{
			// get the list of regapps defined in the source drawing
			if (pDwg->GetTableList(tAr, AcMap::kRegappTable) == AcMap::kOk)
			{
				acutPrintf("\nRegapps:") ;
			}
		}
        else if (!strcmp(res, "Odd"))
		{
			// get the list of Object Data tables defined in the source drawing
			if (pDwg->GetTableList(tAr, AcMap::kODD) == AcMap::kOk)
			{
				acutPrintf("\nObject Data Definitions:") ;
			}
		}
        else if (!strcmp(res, "ASE"))
		{
			// get the list of Link Templates defined in the source drawing
			if (pDwg->GetTableList(tAr, AcMap::kLinkTemplate) == AcMap::kOk)
			{
				acutPrintf("\nLink Path Names:") ;
			}
		}
        else if (!strcmp(res, "textStyle"))
		{
			// get the list of text styles defined in the source drawing
			if (pDwg->GetTableList(tAr, AcMap::kStyleTable) == AcMap::kOk)
			{
				acutPrintf("\nText Styles:") ;
			}
		}

		for (int i = 0; i < tAr.Length(); i++)
		{
			acutPrintf ("\n%s", tAr[i]) ;
		}
   	}
}

// edit transformation settings for the attached drawing
void
editSettings(AcMapAttachedDrawing *pDwg)
{
    char				res[32] ;   
	int					ret = RTNORM ;
	AcMapStringArray	tAr ;
    const char			*path ,
		                *proj ;
	double				angle,
						scale ;
	AcGeVector2d 		offset ;

	// gets the actual system path of the drawing file (don't missmatch with alias path)
    pDwg->GetActualPath(path) ;

	do
	{
		acutPrintf("\n*******Drawing: %s", path) ;

		// retrieve and print projection code
		pDwg->GetProjection(proj) ;
		if (proj && *proj != EOS)
		{
			acutPrintf("\nProjection: \t%s", proj) ;
		}
		if (pDwg->IsTransformationEnabled())
		{
			// retrieve and print current settings
			pDwg->GetRotation(angle) ;
			pDwg->GetScale(scale) ;
			pDwg->GetOffset(offset) ;
			acutPrintf("\nOffset: \t%f %f", offset[0], offset[1]) ;
			acutPrintf("\nRotation: \t%f", angle) ;
			acutPrintf("\nScale: \t%f", scale) ;
		}
	
		if (!pDwg->IsTransformationEnabled())
		{
			acedInitGet(0, "eXit Assign Enable") ;
			ret = acedGetKword("\nAssign Projection/Enable transformation/<eXit>: ", res) ;
		}
		else
		{
			acedInitGet(0, "eXit Rotation Scale Offset Disable") ;
			ret = acedGetKword("\nSet Rotation/Scale/Offset/Disable transformation/<eXit>: ", res) ;
		}

		if (ret == RTNORM)
		{
			if (!strcmp(res, "eXit"))
			{
				break ;
			}
			else if (!strcmp(res, "Enable"))
			{
				// set simple transformation enabled
				pDwg->EnableTransformation(Adesk::kTrue) ;
			}
			else if (!strcmp(res, "Disable"))
			{
				// set simple transformation disabled
				pDwg->EnableTransformation(Adesk::kFalse) ;
			}
			else if (!strcmp(res, "Scale"))
			{
				char buff[128] ;
				sprintf (buff, "\nEnter scale value <%f>: ", scale) ;
				if (acedGetReal(buff, &scale) == RTNORM)
				{
					// set scale
					pDwg->SetScale(scale) ;
				}
			}
			else if (!strcmp(res, "Rotation"))
			{
				char buff[128] ;
				sprintf (buff, "\nEnter rotation value <%f>: ", angle) ;
				if (acedGetReal(buff, &angle) == RTNORM)
				{
					// set rotation
					pDwg->SetRotation(angle) ;
				}
			}
			else if (!strcmp(res, "Offset"))
			{
				ads_point p, p0={0.,0.,0.} ;
				if (acedGetPoint(p0, "\nSelect offset: ", p) == RTNORM)
				{
					// set offset
					offset[0] = p[0] ;
					offset[1] = p[1] ;
					pDwg->SetOffset(offset) ;
				}
			}
			else if (!strcmp(res, "Assign"))
			{
				char buff[128] ;
				sprintf (buff, "\nEnter projection <%s>: ", proj) ;
				if (acedGetString(1, buff, res) == RTNORM)
				{
					// lock the source drawing for write
					if (pDwg->LockForWrite(Adesk::kTrue) == AcMap::kOk)
					{
						// assign the projection code
						if(pDwg->SetProjection(res) == AcMap::kOk)
						{
							// save the database
							pDwg->Save("\nStore projection...") ;
						}
						
						// remove the write lock
						pDwg->UnlockForWrite() ;						
					}
				}
			}
		}
		else
		{
			break ;
		}
        printErrStack() ;

	} while(1) ;
}

// The number of aliases
void editAliases()
{
    AcMapSession *mapApi = NULL ;
    AcMapAliases *pAliases = NULL ;
    char         res[32] ;   
   
    do {
		// get pointer to the root API object
        mapApi = AcMapGetSession();
        if (mapApi == NULL) 
        {
			acutPrintf ("\nCan't connect to MAP") ;
            break ;
        }
        
		// get pointer to the alias list
		if (mapApi->GetAliases(pAliases) == Adesk::kFalse)
        {
            acutPrintf ("\nCan't get aliases") ;
            break ;
        }
        
        do {
			// print the current alias list
            printAliases(pAliases) ;
            acutPrintf ("\n%d Aliases defined", pAliases->CountAliases()) ;

			*res = EOS ;
            acedInitGet(0, "eXit remoVe Update removeAll aDd Rename") ; 
            if (acedGetKword ("\nremoVe/Update/removeAll/aDd/Rename/<eXit>: ", res) == RTNORM)
            {
                if (*res == EOS || !(strcmp(res, "eXit")))
                {
                    break ;
                }
                else if (!(strcmp(res, "remoVe")))
                {
                    *res = EOS ;
                    acedGetString (1, "\nSpecify alias to remove: ", res) ;
                    if (*res != EOS)
                    {
						// remove the alias from the list
                        pAliases->RemoveAlias(res) ;
                    }
                } 
                else if (!(strcmp(res, "removeAll")))
                {
					// remove all aliases from the list
                    pAliases->Clear() ;
                }
                else if (!(strcmp(res, "aDd")))
                {
                    char path[_MAX_PATH] ;
                    acedGetString(1, "\nSpecify alias name: ", res) ;
                    acedGetString(1, "\nSpecify the associated dir: ", path) ;

					// add alias to the list
                    if (pAliases->AddAlias(res, path) != AcMap::kOk)
					{
						AcMapErrorStack *pErr = NULL ;
						if (mapApi->GetErrorStack(pErr))
						{
							pErr->Push(AcMap::kErr, AcMap::kAdeError, "\nBad alias sintax") ;

							AcMapErrorEntry *pEntry = NULL ;
							if (pErr->GetEntry(pEntry, 0))
							{
								pEntry->AddParameter(1, res, path, 0) ;
								pEntry->AddParameter(1, "aaa", "bbb", 1) ;
								delete pEntry ;
							}
						}
					}
                }
                else if (!(strcmp(res, "Update")))
                {
                    *res = EOS ;
                    acedGetString(1, "\nSpecify alias name: ", res) ;
                    if (*res != EOS)
                    {
						// initialize AcMapDriveAlias pointer with NULL to let it be allocated
						// by GetAlias()
                        AcMapDriveAlias *pObj = NULL ;
                        if (pAliases->GetAlias(pObj, res) == AcMap::kOk)
                        {
                            char *pOldPath ;
                            char path[_MAX_PATH] ;
                            
                            pObj->GetPath(pOldPath) ;
                            acutPrintf ("\nOld path %s", pOldPath) ;
                            delete[] pOldPath;
                            acedGetString(1, "\nSpecify the new path: ", path) ;
                            pObj->SetPath(path) ;
                            
                            const char *pName, *pPath ;
                            pObj->GetName(pName) ;
                            pObj->GetPath(pPath) ;
                            acutPrintf ("\nUpdated alias %s %s", pName, pPath) ;

							// don't forget to delete the object
                            delete pObj ;
                        }
                        else
                        {
                            acutPrintf ("\nAlias not found.") ;    
                        }
                    }
                }
                else if (!(strcmp(res, "Rename")))
                {
                    *res = EOS ;
                    acedGetString(1, "\nSpecify alias name: ", res) ;
                    if (*res != EOS)
                    {
                        // initialize AcMapDriveAlias pointer with NULL to let it be allocated
						// by GetAlias()
                        AcMapDriveAlias *pObj = NULL ;
                        if (pAliases->GetAlias(pObj, res) == AcMap::kOk)
                        {
                            char *pOldPath ;
                            char name[32] ;
                            
                            pObj->GetPath(pOldPath) ;
                            acutPrintf ("\nOld path %s", pOldPath) ;
                            delete[] pOldPath;
                            acedGetString(1, "\nSpecify the new name: ", name) ;
                            pObj->SetName(name) ;
                            
                            const char *pName, *pPath ;
                            pObj->GetName(pName) ;
                            pObj->GetPath(pPath) ;
                            acutPrintf ("\nUpdated alias %s %s", pName, pPath) ;

							// don't forget to delete the object
                            delete pObj ;
                        }
                        else
                        {
                            acutPrintf ("\nAlias not found.") ;    
                        }
                    }
                }
            } 
            else
            {
                break ;
            }
            printErrStack() ;
        } while (1) ;
        
    } while (0) ;
   
}

// allocate the query object
AcMapQuery*
getQuery(Adesk::Boolean isCurrent)
{
    AcMapSession	*mapApi = NULL ;
	AcMapProject	*pProj = NULL ;
    AcMapQuery		*pQuery = NULL ;
   
    do {
		// get pointer to the root API object
        mapApi = AcMapGetSession();
        if (mapApi == NULL) 
        {
            acutPrintf ("\nCan't connect to MAP") ;
            break ;
        }
        
		// get the project associated with the current AutoCAD document
		if (mapApi->GetProject(pProj) == Adesk::kFalse)
		{
		    acutPrintf ("\nCan't get MAP project") ;
            break ;
        }
        
		// allocate a query object - it should be deleted by the calling code
		if (pProj->CreateQuery(pQuery, isCurrent) != AcMap::kOk)
		{
            acutPrintf ("\nCan't create query object.") ;
            break ;
        }

	} while (0) ;

	return pQuery ;
}

// get data query type from the user
Adesk::Boolean
getDataQueryType(AcMap::EDataQueryType &type)
{
	char res [132] ;
	Adesk::Boolean ret = Adesk::kFalse ;

	acedInitGet(1,"Ird Ase aTtribute EED") ; 
	if (acedGetKword ("\nIrd/Ase/aTtribute/EED: ", res) == RTNORM)
	{
		ret = Adesk::kTrue ;
		if (!(stricmp(res, "Ird")))
		{
			type = AcMap::kDataIRD ;
		}
		else if (!(stricmp(res, "Ase")))
		{
			type = AcMap::kDataLinkTemplate ;
		}
		else if (!(stricmp(res, "aTtribute")))
		{
			type = AcMap::kDataAttribute ;
		}
		else if (!(stricmp(res, "EED")))
		{
			type = AcMap::kDataEED ;
		}
	}

	return ret ;
}

// get property query type from the user
Adesk::Boolean
getPropertyType(AcMap::EPropertyType &type)
{
	char res [132] ;
	Adesk::Boolean ret = Adesk::kFalse ;
	acedInitGet(1,"Area Block Color Elevation Type Group Layer lEngth lInetype Style Tvalue tHickness") ;
	if (acedGetKword ("\nArea/Block/Color/Elevation/Type/Group/Layer/lEngth/lInetype/\nStyle/Tvalue/tHickness: ", res) == RTNORM)
	{
		ret = Adesk::kTrue ;

		if (!(stricmp(res, "Area")))
		{
			type = AcMap::kArea ;
		}
		else if (!(stricmp(res, "Block")))
		{
			type = AcMap::kBlockName ;
		}
		else if (!(stricmp(res, "Color")))
		{
			type = AcMap::kColor ;
		}
		else if (!(stricmp(res, "Elevation")))
		{
			type = AcMap::kElevation ;
		}
		else if (!(stricmp(res, "Type")))
		{
			type = AcMap::kEntType ;
		}
		else if (!(stricmp(res, "Group")))
		{
			type = AcMap::kGroup ;
		}
		else if (!(stricmp(res, "Layer")))
		{
			type = AcMap::kLayer ;
		}
		else if (!(stricmp(res, "lEngth")))
		{
			type = AcMap::kLength ;
		}
		else if (!(stricmp(res, "lInetype")))
		{
			type = AcMap::kLineType ;
		}
		else if (!(stricmp(res, "Style")))
		{
			type = AcMap::kTextStyle ;
		}
		else if (!(stricmp(res, "Tvalue")))
		{
			type = AcMap::kTextValue ;
		}
		else if (!(stricmp(res, "tHickness")))
		{
			type = AcMap::kThickness ;
		}
	}

	return ret ;
}

// get query condition operator from the user
Adesk::Boolean
getCondition(AcMap::EConditionOperator &op)
{
	char res [132] ;
	Adesk::Boolean ret = Adesk::kFalse ;
	acedInitGet(1,"Unequal Less LessEqual Equal GreaterEqual Greater") ; 
	if (acedGetKword ("\nUnequal/Less/LessEqual/Equal/GreaterEqual/Greater: ", res) == RTNORM)
	{
		ret = Adesk::kTrue ;
		if (!(stricmp(res, "Unequeal")))
		{
			op = AcMap::kCondNotEq ;
		}
		else if (!(stricmp(res, "Equal")))
		{
			op = AcMap::kCondEq ;
		}
		else if (!(stricmp(res, "Greate")))
		{
			op = AcMap::kCondGT ;
		}
		else if (!(stricmp(res, "GreateEqual")))
		{
			op = AcMap::kCondGTorEq ;
		}
		else if (!(stricmp(res, "Less")))
		{
			op = AcMap::kCondLT ;
		}
		else if (!(stricmp(res, "LessEqual")))
		{
			op = AcMap::kCondLTorEq ;
		}
	}

	return ret ;
}

// get range operator
Adesk::Boolean
GetRangeLineDef(
	AcMap::ERangeOperator &kOp,
	char *pcDef,
	char *pcRet
)
{
	Adesk::Boolean ret = Adesk::kFalse ;
	char buff[132] ;

	acedInitGet (1, "EQ GT GE LT LE NE Otherwise") ;
	if (acedGetKword ("\nEQ/GT/GE/LT/LE/NE/Otherwise: ", buff) == RTNORM)
	{
		if (!stricmp(buff, "EQ"))
		{
			kOp = AcMap::kRangeEq ;
		}
		else if (!stricmp(buff, "GT"))
		{
			kOp = AcMap::kRangeGT ;
		}
		else if (!stricmp(buff, "GE"))
		{
			kOp = AcMap::kRangeGTorEq ;
		}
		else if (!stricmp(buff, "LT"))
		{
			kOp = AcMap::kRangeLT ;
		}
		else if (!stricmp(buff, "LE"))
		{
			kOp = AcMap::kRangeLTorEq ;
		}
		else if (!stricmp(buff, "NE"))
		{
			kOp = AcMap::kRangeNotEq ;
		}
		else if (!stricmp(buff, "Otherwise"))
		{
			kOp = AcMap::kRangeOtherwise ;
		}

		*pcRet = EOS ;
		if (acedGetString (1, "\nEnter range definition: ", pcDef) == RTNORM 
			&& acedGetString (1, "\nEnter return value: ", pcRet) == RTNORM
			&& *pcRet != EOS)
		{
			ret = Adesk::kTrue ;
		}
	}

	return ret ;
}

// get query join operation from the user
Adesk::Boolean
JoinOperation(AcMap::EJoinOperator &kOp)
{
	char res[132] ;
	int	 ret ;
	Adesk::Boolean retCode = Adesk::kFalse ;
	
	acedInitGet(0, "And Or") ;
	ret = acedGetKword("\nAnd/Or: ", res) ;
	if (ret == RTNORM)
	{
		retCode = Adesk::kTrue ;
		if (!stricmp(res, "And"))
		{
			kOp = AcMap::kOperatorAnd ;
		}
		else 
		{
			kOp = AcMap::kOperatorOr ;
		}
	}

	return retCode ;
}

// define point location boundary
AcMapPointBoundary*
definePoint()
{
	AcMapPointBoundary *pBoundary = NULL ;
	ads_point p ;

	if (acedGetPoint (NULL, "\nSelect point: ", p) == RTNORM)
	{
		AcGePoint3d pp(p[0], p[1], 0.0) ;
		pBoundary = new AcMapPointBoundary(pp) ;
	}

	return pBoundary ;
}

// define circle location boundary
AcMapCircleBoundary*
defineCircle()
{
	AcMapCircleBoundary *pBoundary = NULL ;
	ads_point p ;
	double rad ;

	if (acedGetPoint (NULL, "\nSelect point: ", p) == RTNORM &&
		acedGetDist(p, "\nSelect radius: ", &rad) == RTNORM)
	{
		AcGePoint3d pp(p[0], p[1], 0.0) ;
		pBoundary = new AcMapCircleBoundary(pp, rad) ;
	}

	return pBoundary ;
}

// define fence location boundary
AcMapFenceBoundary*
defineFence()
{
	AcMapFenceBoundary *pBoundary = NULL ;
	Adesk::Boolean isFirst = Adesk::kTrue ;
	ads_point p1, p2 ;
	int	ret = RTNORM ;
	AcGePoint3dArray pp ;

	do {
		ret = acedGetPoint (isFirst? NULL : p1, "\nSelect point: ", p2) ;
		if (ret == RTNORM)
		{
			pp.append(AcGePoint3d(p2[0], p2[1], 0.0)) ;
			p1[0] = p2[0] ;
			p1[1] = p2[1] ;
			p1[2] = 0.0;
			isFirst = Adesk::kFalse ;
		} else if (ret == RTNONE && pp.length() > 0) 
		{
			pBoundary = new AcMapFenceBoundary(pp) ;
			break ;
		}

	} while (ret == RTNORM) ;

	return pBoundary ;
}

// define buffer fence location boundary
AcMapBufferFenceBoundary*
defineBufferFence()
{
	AcMapBufferFenceBoundary *pBoundary = NULL ;
	Adesk::Boolean isFirst = Adesk::kTrue ;
	ads_point p1, p2 ;
	int	ret = RTNORM ;
	AcGePoint3dArray pp ;
	double dWidth ;

	do {
		ret = acedGetPoint (isFirst? NULL : p1, "\nSelect point: ", p2) ;
		if (ret == RTNORM)
		{
			pp.append(AcGePoint3d(p2[0], p2[1], 0.0)) ;
			p1[0] = p2[0] ;
			p1[1] = p2[1] ;
			p1[2] = 0.0;
			isFirst = Adesk::kFalse ;
		} 
		else if (ret == RTNONE && pp.length() > 0) 
		{
			if (acedGetDist(p2, "\nSelect buffer width: ", &dWidth) == RTNORM)
			{
				pBoundary = new AcMapBufferFenceBoundary(pp, dWidth) ;
			}
		}

	} while (ret == RTNORM) ;

	return pBoundary ;
}

// define polygon location boundary
AcMapPolygonBoundary*
definePolygon()
{
	AcMapPolygonBoundary *pBoundary = NULL ;
	Adesk::Boolean isFirst = Adesk::kTrue ;
	ads_point p1, p2 ;
	int	ret = RTNORM ;
	AcGePoint3dArray pp ;

	do {
		ret = acedGetPoint (isFirst? NULL : p1, "\nSelect point: ", p2) ;
		if (ret == RTNORM)
		{
			pp.append(AcGePoint3d(p2[0], p2[1], 0.0)) ;
			p1[0] = p2[0] ;
			p1[1] = p2[1] ;
			p1[2] = 0.0;
			isFirst = Adesk::kFalse ;
		} else if (ret == RTNONE && pp.length() > 0) 
		{
			pBoundary = new AcMapPolygonBoundary(pp) ;
		}

	} while (ret == RTNORM) ;

	return pBoundary ;
}

// define window location boundary
AcMapWindowBoundary*
defineWindow()
{
	AcMapWindowBoundary *pBoundary = NULL ;
	ads_point p1, p2 ;

	if (acedGetPoint (NULL, "\nSelect the first point: ", p1) == RTNORM
		&& acedGetPoint (p1, "\nSelect the second point: ", p2) == RTNORM)
	{
		pBoundary = new AcMapWindowBoundary(AcGePoint3d(p1[0], p1[1], 0.), AcGePoint3d(p2[0], p2[1], 0.)) ;
	} 

	return pBoundary ;
}

// define polyline location boundary
AcMapPolylineBoundary*
definePolyline()
{
	AcMapPolylineBoundary *pBoundary = NULL ;
	ads_name ename ;
	ads_point p ;
	AcDbObjectId	Id ;

	if (acedEntSel ("\nSelect polyline, line or arc: ", ename, p) == RTNORM 
		&& acdbGetObjectId(Id, ename) == Acad::eOk)
	{
		pBoundary = new AcMapPolylineBoundary ;
		if (pBoundary->Init(Id) != AcMap::kOk)
		{
			delete pBoundary ;
			pBoundary = NULL ;
		}
	} 

	return pBoundary ;
}

// define buffer polyline location boundary
AcMapBufferPolylineBoundary*
defineBufferPolyline()
{
	AcMapBufferPolylineBoundary *pBoundary = NULL ;
	ads_name ename ;
	ads_point p ;
	AcDbObjectId	Id ;
	double dWidth ;

	if (acedEntSel ("\nSelect polyline, line or arc: ", ename, p) == RTNORM 
		&& acdbGetObjectId(Id, ename) == Acad::eOk
		&& acedGetDist(NULL, "\nSelect buffer width: ", &dWidth) == RTNORM)
	{
		pBoundary = new AcMapBufferPolylineBoundary ;
		if (pBoundary->Init(Id, dWidth) != AcMap::kOk)
		{
			delete pBoundary ;
			pBoundary = NULL ;
		}
	} 

	return pBoundary ;
}

// define closed polyline location boundary
AcMapClosedPolylineBoundary*
defineClosedPolyline()
{
	AcMapClosedPolylineBoundary *pBoundary = NULL ;
	ads_name ename ;
	ads_point p ;
	AcDbObjectId	Id ;

	if (acedEntSel ("\nSelect polyline or arc: ", ename, p) == RTNORM 
		&& acdbGetObjectId(Id, ename) == Acad::eOk)
	{
		pBoundary = new AcMapClosedPolylineBoundary ;
		if (pBoundary->Init(Id) != AcMap::kOk)
		{
			delete pBoundary ;
			pBoundary = NULL ;
		}
	} 

	return pBoundary ;
}

// get location type from the user
Adesk::Boolean
getLocationType(AcMap::ELocationType &type)
{
	int retCode ;
	Adesk::Boolean ret = Adesk::kFalse ;
	char res[132] ;

	acedInitGet (0, "Inside Crossing") ;
	retCode = acedGetKword("\nInside/<Crossing>: ", res) ;
	if (retCode != RTCAN)
	{
	 	if (retCode == RTNORM && !(stricmp(res, "Inside")))
	 	{
	 		type = AcMap::kLocationInside ;
	 	}
	 	else
	 	{
	 		type = AcMap::kLocationCrossing ;
	 	}
		ret = Adesk::kTrue ;
	}

	return ret ;
}

// define location condition
AcMapLocationCondition*
defineLocationQuery()
{
	AcMapLocationCondition *pCond = NULL ;
	AcMapLocationBoundary *pBound = NULL ;
	int retCode ;
	char res[132] ;
	AcMap::ELocationType type ;

	acedInitGet(0, "Point Circle Fence Bufferfence polyGon Window polYline All") ;
	retCode = acedGetKword("\nPoint/Circle/Fence/Bufferfence/polyGon/Window/polYline/<All>: ", res) ;
	if (retCode == RTNORM)
	{
		if (!stricmp(res, "All"))
		{
			// allocate new location condition object
			pCond = new AcMapLocationCondition ;

			// set ALL condition
			pCond->SetBoundary(&AcMapAllBoundary()) ;
		}
		else if (!stricmp(res, "Point"))
		{
			// define POINT boundary - nested dialog
			pBound = definePoint() ;
			if (pBound != NULL)
			{
				// allocate new location condition object
				pCond = new AcMapLocationCondition ;

				// initialize POINT condition
				pCond->SetBoundary(pBound) ;

				// the only type of location type can be applied to the POINT condition
				pCond->SetLocationType(AcMap::kLocationCrossing) ;

			}
		}
		else if (!stricmp(res, "Circle"))
		{
			// define CIRCLE boundary - nested dialog
			pBound = defineCircle() ;
			if (pBound != NULL)
			{
				// define location type - nested dialog
				if (getLocationType(type))
				{
					// allocate new location condition object
					pCond = new AcMapLocationCondition ;

					// initialize CIRCLE condition
					pCond->SetLocationType(type) ;
					pCond->SetBoundary(pBound) ;
				}
			}
		}
		else if (!stricmp(res, "Fence"))
		{
			// define FENCE boundary - nested dialog
			pBound = defineFence() ;
			if (pBound != NULL)
			{
				// allocate new location condition object
				pCond = new AcMapLocationCondition ;

				// initialize FENCE condition
				pCond->SetBoundary(pBound) ;

				// the only type of location type can be applied to the FENCE condition
				pCond->SetLocationType(AcMap::kLocationCrossing) ;
			}
		}
		else if (!stricmp(res, "Bufferfence"))
		{
			// define BUFFER FENCE boundary - nested dialog
			pBound = defineBufferFence() ;
			if (pBound != NULL)
			{
				// get location type - nested dialog
				if (getLocationType(type))
				{
					// allocate new location condition object
					pCond = new AcMapLocationCondition ;

					// initialize BUFFER FENCE condition
					pCond->SetLocationType(type) ;
					pCond->SetBoundary(pBound) ;
				}
			}
		}
		else if (!stricmp(res, "Window"))
		{
			// define WINDOW boundary - nested dialog
			pBound = defineWindow() ;
			if (pBound != NULL)
			{
				// get location type - nested dialog
				if (getLocationType(type))
				{
					// allocate new location condition
					pCond = new AcMapLocationCondition ;

					// initialize WINDOW condition
					pCond->SetLocationType(type) ;
					pCond->SetBoundary(pBound) ;
				}
			}
		}
		else if (!stricmp(res, "Polygon"))
		{
			// define POLYGON boundary - nested dialog
			pBound = definePolygon() ;
			if (pBound != NULL)
			{
				// get location type - nested dialog
				if (getLocationType(type))
				{
					// allocate new location condition
					pCond = new AcMapLocationCondition ;

					// initialize POLYGON condition
					pCond->SetLocationType(type) ;
					pCond->SetBoundary(pBound) ;
				}
			}
		}
		else if (!stricmp(res, "Polyline"))
		{
			acedInitGet (0, "Polyline Buffer polyGon") ;
			retCode = acedGetKword("\nPolyline/Buffer/polyGon: ", res) ;
			if (retCode == RTNORM)
			{
				if (!stricmp(res, "Polyline"))
				{
					// define POLYLINE FENCE boundary - nested dialog
					pBound = definePolyline() ;
					if (pBound != NULL)
					{
						// allocate new location condition
						pCond = new AcMapLocationCondition ;

						// initialize POLYLINE condition
						pCond->SetBoundary(pBound) ;

						// the only type of location type can be applied to the POLYLINE condition
						pCond->SetLocationType(AcMap::kLocationCrossing) ;
					}
				}
				else if (!stricmp(res, "polyGon"))
				{
					// initialize POLYLINE POLYGON boundary - nested dialog
					pBound = defineClosedPolyline() ;
					if (pBound != NULL)
					{
						// get location type - nested dialog
						if (getLocationType(type))
						{
							// allocate new location condition 
							pCond = new AcMapLocationCondition ;

							// initialize POLYLINE POLYGON condition
							pCond->SetLocationType(type) ;
							pCond->SetBoundary(pBound) ;
						}
					}
				}
				else if (!stricmp(res, "Buffer"))
				{
					// define POLYLINE BUFFER FENCE boundary - nested dialog
					pBound = defineBufferPolyline() ;
					if (pBound != NULL)
					{
						// get location type - nested dialog
						if (getLocationType(type))
						{
							// allocate new location condition
							pCond = new AcMapLocationCondition ;

							// initialize POLYLINE BUFFER FENCE condition
							pCond->SetLocationType(type) ;
							pCond->SetBoundary(pBound) ;
						}
					}
				}
			}

		}

	}
	else if (retCode == RTNONE)
	{
		// allocate new location condition
		pCond = new AcMapLocationCondition ;

		// initialize ALL condition
		pCond->SetBoundary(&AcMapAllBoundary()) ;
	}

	// the condition copies the boundary object
	delete pBound ;

	return pCond ;
}

// define property condition
AcMapPropertyCondition*
definePropertyQuery()
{

	AcMapPropertyCondition *pCond ;

	do
	{
		AcMap::EConditionOperator op ;
		AcMap::EPropertyType type ;
		char value[132] ;

		// get property type and condition - nested dialogs
		if (!getPropertyType(type) ||!getCondition(op))
		{
			break ;
		}

		if (acedGetString(0, "\nEnter value: ",value) != RTNORM)
		{
			break ;
		}

		// allocate new property condition object
		pCond = new AcMapPropertyCondition(AcMap::kOperatorAnd, type, op, value) ;

	} while (0) ;

	return pCond ;
}

// define data condition
AcMapDataCondition*
defineDataQuery()
{
	AcMapDataCondition *pCond = NULL ;

	do
	{
		AcMap::EConditionOperator op ;
		AcMap::EDataQueryType type ;
		char pcTable[132],
			 pcField[132],
			 pcValue[132];

		// get data type and condition - nested dialogs
		if (!getDataQueryType(type) || !getCondition(op))
		{
			break ;
		}

		if (acedGetString(0, "\nEnter table name: ", pcTable) != RTNORM)
		{
			break ;
		}

		if (acedGetString(0, "\nEnter field name: ", pcField) != RTNORM)
		{
			break ;
		}

		if (acedGetString(0, "\nEnter value: ", pcValue) != RTNORM)
		{
			break ;
		}

		// allocate new data condition object
		pCond = new AcMapDataCondition(AcMap::kOperatorAnd, type, op, pcTable, pcField, pcValue) ;

	} while (0) ;

	return pCond ;
}

// define query branch
Adesk::Boolean
define(AcMapQueryBranch &tBranch)
{
	char res[132] ;
	int	 ret ;
	
	do 
	{
		*res = '\0' ;
		acedInitGet(0, "Group Location Property Data Sql eXit") ;
		ret = acedGetKword("\nGroup/Location/Property/Data/Sql/<eXit>: ", res) ;

		if (!stricmp(res, "Group"))
		{
			AcMapQueryBranch tNestBranch ;

			// define nested query branch - nested dialog
			define(tNestBranch) ;

			// be sure branch is not empty
			if (tNestBranch.CountOperands() == 0)
			{
				acutPrintf("\nInvalid group.") ;
			}
			else
			{
				// get join operator - nested dialog
				AcMap::EJoinOperator kOp ;
				JoinOperation(kOp) ;

				// add nested branch to the current branch
				tNestBranch.SetJoinOperator(kOp) ;
				tBranch.AppendOperand(&tNestBranch) ;
			}
		}
		else if (!stricmp(res, "Location"))
		{
			// define location condition - nested dialog
			AcMapLocationCondition *pCond = defineLocationQuery() ;
			if (pCond)
			{
				if (tBranch.CountOperands() > 0)
				{
					// get join operation - nested dialog
					AcMap::EJoinOperator kOp ;
					if (!JoinOperation(kOp))
					{
						//nested dialog is canceled
						delete pCond ;
						continue ;
					}

					// add the condition to the branch
					pCond->SetJoinOperator(kOp) ;
					tBranch.AppendOperand(pCond) ;
				}
				delete pCond ;
			}
		}
		else if (!stricmp(res, "Property"))
		{
			// define property condition - nested dialog
			AcMapPropertyCondition *pCond = definePropertyQuery() ;
			if (pCond)
			{
				if (tBranch.CountOperands() > 0)
				{
					// get join operation
					AcMap::EJoinOperator kOp ;
					if (!JoinOperation(kOp))
					{
						//nested dialog is canceled
						delete pCond ;
						continue ;
					}

					// add the condition to the branch
					pCond->SetJoinOperator(kOp) ;
					tBranch.AppendOperand(pCond) ;
				}
				delete pCond ;
			}
		}
		else if (!stricmp(res, "Data"))
		{
			// define data condition - nested dailog
			AcMapDataCondition *pCond = defineDataQuery() ;
			if (pCond)
			{
				if (tBranch.CountOperands() > 0)
				{
					// get join operation - nested dialog
					AcMap::EJoinOperator kOp ;
					if (!JoinOperation(kOp))
					{
						//nested dialog is canceled
						delete pCond ;
						continue ;
					}

					// add the condition to th branch
					pCond->SetJoinOperator(kOp) ;
					tBranch.AppendOperand(pCond) ;
				}
				delete pCond ;
			}
		}
		else if (!stricmp(res, "eXit"))
		{
			break ;
		}
		
	} while (ret == RTNORM) ;

	return Adesk::kTrue ;
}

// save the query
void
saveQuery(AcMapQuery *pQuery)
{
	if (pQuery)
	{
		char res [128] ;
		acedInitGet (1, "Externally Internally") ;
		if (acedGetKword("\nExternally/Internally: ", res) == RTNORM)
		{
			if (!stricmp(res, "Externally"))
			{
				*res = EOS ;
				if (acedGetString (1, "\nFile name: ", res) == RTNORM && *res != EOS)
				{
					// save the query to the specified file
					pQuery->Save(res, AcMap::kSaveDrawingSet | AcMap::kSaveLocationCoordinates
									  | AcMap::kSavePropertyAlteration | AcMap::kAutoExecute) ;			

				}
			}
			else
			{
				char pcCat[128],
					 pcName[128],
					 pcFileName[128] ;
				*pcCat = *pcName = *pcFileName = EOS ;
				if (acedGetString (1, "\nCategory name: ", pcCat) == RTNORM 
					&& acedGetString (1, "\nQuery name: ", pcName) == RTNORM 
					&& acedGetString (0, "\nFile name: ", pcFileName) == RTNORM) 
				{
					if (*pcFileName == EOS)
					{
						// save the query to the query library
						pQuery->Save(pcCat, pcName) ;
					}
					else
					{
						// save the query to the query library as the reference to the external
						// file
						pQuery->Save(pcCat, pcName, NULL, pcFileName, AcMap::kSaveDrawingSet 
							| AcMap::kSaveLocationCoordinates | AcMap::kSavePropertyAlteration
							| AcMap::kAutoExecute) ;
					}
				}
			}
		}
	}
}

// load the query
void
load(AcMapQuery *pQuery)
{
	if (pQuery)
	{
		char pcCat[128],
			 pcName[128] ;
		*pcName = *pcCat = EOS ;
		if (acedGetString (1, "\nCategory name: ", pcCat) == RTNORM 
			&& acedGetString (0, "\nQuery name: ", pcName) == RTNORM)
		{
			if (*pcName != EOS && *pcCat != EOS)
			{
				// load the query from the query library
				pQuery->Load(pcCat, pcName) ;
			}
			else
			{
				// load the query from the external file
				pQuery->Load(pcCat) ;
			}
		}
	}
}

// print the report template
void		
printTemplate(AcMapReportTemplate *pTempl) 
{
	if (pTempl)
	{
		// initialize AcMapTemplateLine to NULL
		AcMapTemplateLine *pLine = NULL ;
		int count = pTempl->CountLines() ;
		if (count)
		{
			acutPrintf ("\nReport Template:") ;
			for (int i = 0; i < count; i++)
			{
				// be sure pLine is initialized to NULL to allocate the object at the first call
				if (pTempl->GetLine(pLine, i) == AcMap::kOk)
				{
					AcMapExpression *pExpr ;
					const char *pExprStr ;
					// get the associated expression object
					if (pLine->GetExpression(pExpr) == AcMap::kOk)
					{
						// represent the expression as a string
						pExpr->GetExpressionString(pExprStr) ;
						acutPrintf ("\n\t%s", pExprStr) ;

						// don't forget to delete the object
						delete pExpr ;
					}
				}
			}

			// don't forget to delete the object
			delete pLine ;

			const char *pcFile ;
			pTempl->GetFileName(pcFile) ;
			acutPrintf ("\nTo File %s", pcFile) ;
		}
	}
}

// define report template
void		
DefineTemplate(AcMapReportTemplate *pTempl) 
{
	if (pTempl)
	{
		char pcBuf[132] ;
		do {
			// print the current template
			printTemplate(pTempl) ;

			acedInitGet (0, "Add Remove Update eXit") ;
			if (acedGetKword ("\nAdd/Remove/Update: ", pcBuf) == RTNORM)
			{
				if (!strcmp(pcBuf, "Add"))
				{
					*pcBuf = EOS ;
					if (acedGetString (1, "Enter expression: ", pcBuf) == RTNORM
						&& *pcBuf != EOS)
					{
						AcMapSession *pSess = AcMapGetSession() ;
						assert (NULL != pSess) ;

						// get the corresponding project
						AcMapProject *pProj = NULL;
						assert (pSess->GetProject(pProj) == Adesk::kTrue) ;

						// allocate new expression
						AcMapExpression *pExpr = NULL ;
						if (pProj->DefineExpression(pExpr, pcBuf) == AcMap::kOk)
						{
							// add expression to the template
							pTempl->AddLine(*pExpr) ;

							// don't forget to delete the object
							delete pExpr ;
						}
					}
				}
				else if (!strcmp(pcBuf, "Remove"))
				{
					int i ;
					if (acedGetInt ("\nEnter line number:", &i) == RTNORM)
					{
						// remove the template line by index
						pTempl->RemoveLine(i) ;
					}
				}
				else if (!strcmp(pcBuf, "Update"))
				{
					int i ;
					*pcBuf = EOS ;
					if (acedGetInt ("\nEnter line number:", &i) == RTNORM
					    && acedGetString (1, "Enter expression: ", pcBuf) == RTNORM
						&& *pcBuf != EOS)
					{
						// initialize AcMapTemplateLine pointer to NULL
						AcMapTemplateLine *pLine = NULL ;
						if (pTempl->GetLine(pLine, i) == AcMap::kOk)
						{
							// update the expression
							pLine->SetExpression(pcBuf) ;
							delete pLine ;
						}
					}
				}
			}
			else
			{
				break ;
			}
		} while (1) ;
	}
}

//define property alteration
void		
DefineAlteration(AcMapPropertyAlterationDefinition *pDef)
{
	int retCode ;

	if (pDef)
	{
		char pcBuf[132] ;
		AcMapPropertyAlteration *pcAlt = NULL;
		
		do {
			acedInitGet (0, "Block Color Layer Rotation Elevation heIght LType Scale Style teXT thickneSs Width TEntity Hatch eXit") ;
			retCode = acedGetKword ("Block/Color/Layer/Rotation/Elevation/heIght/LType/Scale/Style/Text/thickneSs/Width/TEntity/Hatch/<eXit>: ",
					pcBuf) ;

			if (!stricmp(pcBuf, "Block"))
			{	
				// add BLOCK alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationBlockName) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter block name : ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}

					// don't forget to delete the object
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "Layer"))
			{	
				// add LAYER alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationLayer) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter layer name: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "Rotation"))
			{	
				// add ROTATION alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationRotation) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter rotation value: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "Elevation"))
			{	
				// add ELEVATION alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationElevation) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter elevation value: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "Height"))
			{	
				// add HEIGHT alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationHeight) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter height value: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "LType"))
			{	
				// add LINETYPE alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationLineType) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter lynetype name: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "Scale"))
			{	
				// add SCALE alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationScale) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter scale value: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			// add TEXT alteration - don't missmatch with TEXT ENTITY alteration
			else if (!stricmp(pcBuf, "Text"))
			{	
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationText) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter text value: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set the new text value
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "Width"))
			{	
				// add WIDTH alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationWidth) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter width value: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an expression
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "TEntity"))
			{	
				// add TEXT ENTITY alteration - don't missmatch with TEXT alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationTextEntity) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter text value: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set an text value, all other text parameters are default
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}
			else if (!stricmp(pcBuf, "Hatch"))
			{	
				// add HATCH alteration
				if (pDef->AddAlteration(pcAlt, AcMap::kAlterationHatch) == AcMap::kOk)
				{
					*pcBuf = EOS ;
					retCode = acedGetString (1, "Enter hatch patter: ", pcBuf) ;
					if (*pcBuf != EOS)
					{
						// set the pattern, all other parameters are default
						pcAlt->SetExpression(pcBuf) ;
					}
					delete pcAlt ;
				}
			}

			else if (!stricmp(pcBuf, "eXit"))
			{
				break ;
			}
		} while (retCode == RTNORM) ;
	}
}

// print out the query library 
void printQLib()
{
	do
	{
		// get the root API object
		AcMapSession *mapApi = AcMapGetSession();
		if (mapApi == NULL) 
		{
			break ;
		}
	
		// get the project, associated with the current AutoCAD document 
		AcMapProject *pProj ;
		if (!mapApi->GetProject(pProj))
		{
			break ;
		}

		// gets the query library from the project
		AcMapQueryLibrary *pQLib = NULL ;
		if (!pProj->GetQueryLibrary(pQLib))
		{
			break ;
		}

		int iCatQty = pQLib->CountCategories() ;

		if (iCatQty)
		{
			acutPrintf ("\n****** Query Library *****") ;
		}

		// initialize AcMapQueryCategory pointer with NULL before the loop
		AcMapQueryCategory *pQCat = NULL ;
		for (int i = 0; i < iCatQty; i++)
		{
			// pQCat object will be allocated once at the first call
			if (pQLib->GetCategory(pQCat, i) == AcMap::kOk)
			{
				const char *pcName;
				pQCat->GetName(pcName) ;
				acutPrintf("\n*Category %s", pcName) ;

				int iQryQty = pQCat->CountQueries() ;

				// initialize AcMapQueryAttribute pointer with NULL before the loop
				AcMapQueryAttribute *pQAttr = NULL ;
				for (int j = 0; j < iQryQty; j++)
				{
					// pQAttr object will be allocated once at the first call
					if (pQCat->GetQuery(pQAttr, j) == AcMap::kOk)
					{
						const char *pcQryName, *pcQryDsc ;
						pQAttr->GetName(pcQryName) ;
						pQAttr->GetDescription(pcQryDsc) ;

						if (pcQryDsc && *pcQryDsc != EOS)
						{
							acutPrintf("\n\tQuery %s - %s", pcQryName, pcQryDsc) ;
						}
						else
						{
							acutPrintf("\n\tQuery %s", pcQryName) ;
						}
					}
				}
				// don't forget to delete the object after the loop
				delete pQAttr ;
			}
		}

		// don't forget to delete the object after the loop
		delete pQCat ;

	} while (0) ;
}

// add the query to the query library
void addQuery()
{
	AcMapQueryCategory *pCat = NULL ;

	do
	{
		// get the root API object
		AcMapSession *mapApi = AcMapGetSession();
		if (mapApi == NULL) 
		{
			break ;
		}
	
		// get the project, associated with the current AutoCAD document 
		AcMapProject *pProj ;
		if (!mapApi->GetProject(pProj))
		{
			break ;
		}

		// gets the query library from the project
		AcMapQueryLibrary *pQLib = NULL ;
		if (!pProj->GetQueryLibrary(pQLib))
		{
			break ;
		}

		char pcCatName[132] ;
		*pcCatName = EOS ;
		if (acedGetString (1, "\nEnter category name: ", pcCatName) != RTNORM 
			|| *pcCatName == EOS)
		{
			break ;
		}

		// get access to the category or create it if missing
		// Be sure pCat was initialized to NULL to be allocated
		if (pQLib->GetCategory(pCat, pcCatName) != AcMap::kOk)
		{
			if (pQLib->AddCategory(pCat, pcCatName) != AcMap::kOk)
			{
				break ;
			}
		}

		char pcQryName[132] ;
		*pcQryName = EOS ;
		if (acedGetString (1, "\nEnter query name: ", pcQryName) != RTNORM 
			|| *pcQryName == EOS)
		{
	
			break ;
		}

		char pcFileName[132] ;
		*pcFileName = EOS ;
		if (acedGetString (1, "\nEnter file name: ", pcFileName) != RTNORM) 
		{
	
			break ;
		}

		if (*pcFileName == EOS)
		{
			// allocate the interface to the current MAP query 
			AcMapQuery *pQuery = getQuery(Adesk::kTrue) ;

			if(pQuery)
			{
				// add the current query to the library
				if (pCat->AddQuery(pcQryName, NULL, *pQuery) == AcMap::kOk)
				{
					acutPrintf ("\ndone") ;
				}

				// don't forget to delete the object
				delete pQuery ;
			}
		}
		else
		{
			// add query to the library as the reference to the file
			if (pCat->AddQuery(pcQryName, "Bla-Bla-Bla", pcFileName) == AcMap::kOk)
			{
				acutPrintf ("\ndone") ;
			}
		}
	
	} while (0) ;

	if(pCat)
		delete pCat ;
}

// remove the query or the category from the query library
void removeQuery()
{
	AcMapQueryCategory *pCat = NULL ;

	do
	{
		// get the root API object
		AcMapSession *mapApi = AcMapGetSession();
		if (mapApi == NULL) 
		{
			break ;
		}
	
		// get the project, associated with the current AutoCAD document 
		AcMapProject *pProj ;
		if (!mapApi->GetProject(pProj))
		{
			break ;
		}

		// gets the query library from the project
		AcMapQueryLibrary *pQLib = NULL ;
		if (!pProj->GetQueryLibrary(pQLib))
		{
			break ;
		}

		char pcCatName[132] ;
		*pcCatName = EOS ;
		if (acedGetString (1, "\nEnter category name: ", pcCatName) != RTNORM 
			|| *pcCatName == EOS)
		{
			break ;
		}

		char pcQryName[132] ;
		*pcQryName = EOS ;
		if (acedGetString (1, "\nEnter query name: ", pcQryName) != RTNORM) 
		{
	
			break ;
		}

		if (*pcQryName == EOS)
		{
			// remove the specified category
			if (pQLib->RemoveCategory(pcCatName) == AcMap::kOk)
			{
				acutPrintf ("\nDone.") ;
			}
		}
		else
		{
			// get access to the category 
			if (pQLib->GetCategory(pCat, pcCatName) == AcMap::kOk)
			{
				// remove the query
				if (pCat->RemoveQuery(pcQryName) == AcMap::kOk)
				{
					acutPrintf ("\nDone.") ;
				}
			}
		}

	} while(0) ;

	if(pCat)
		delete pCat ;
}

// rename the query or the query category
void renameQuery()
{
	AcMapQueryCategory *pCat = NULL ;

	do
	{
		// get the root API object
		AcMapSession *mapApi = AcMapGetSession();
		if (mapApi == NULL) 
		{
			break ;
		}
	
		// get the project, associated with the current AutoCAD document 
		AcMapProject *pProj ;
		if (!mapApi->GetProject(pProj))
		{
			break ;
		}

		// gets the query library from the project
		AcMapQueryLibrary *pQLib = NULL ;
		if (!pProj->GetQueryLibrary(pQLib))
		{
			break ;
		}

		char pcCatName[132] ;
		*pcCatName = EOS ;
		if (acedGetString (1, "\nEnter category name: ", pcCatName) != RTNORM 
			|| *pcCatName == EOS)
		{
			break ;
		}

		char pcQryName[132] ;
		*pcQryName = EOS ;
		if (acedGetString (1, "\nEnter query name: ", pcQryName) != RTNORM) 
		{
	
			break ;
		}
		
		// gets the specified category, pCat will be allocated
		if (pQLib->GetCategory(pCat, pcCatName) != AcMap::kOk)
		{
			break ;
		}

		char pcNewName[132] ;
		*pcNewName = EOS ;
		if (*pcQryName == EOS)
		{
		    if (acedGetString(1, "\nEnter new category name: ", pcNewName) != RTNORM
			    || *pcNewName == EOS)
			{
			    break ;
			}

			// renames the category
			if (pCat->SetName(pcNewName) == AcMap::kOk)
			{
				acutPrintf ("\nDone.") ;
			}
		}
		else
		{
		    if (acedGetString(1, "\nEnter new query name: ", pcNewName) != RTNORM
			    || *pcNewName == EOS)
			{
			    break ;
			}

			// initialize AcMapQueryAttribute to NULL before calling GetQuery
			AcMapQueryAttribute *pAttr = NULL ;
			if (pCat->GetQuery(pAttr, pcQryName) == AcMap::kOk)
			{
				// rename the query
				if (pAttr->SetName(pcNewName) == AcMap::kOk)
				{
					acutPrintf ("\nDone.") ;
				}
				delete pAttr ;
			}
		}
	} while(0) ;

	delete pCat ;	
}

// print out range library
void
printRangeLib(AcMapRangeLibrary *pLib)
{
	// initialize AcMapRangeTable pointer to NULL 
	AcMapRangeTable *pTab = NULL ;

	int imax = pLib->CountRangeTables() ;
	for (int i = 0; i < imax; i++)
	{
		if (pLib->GetRangeTable(pTab, i) == AcMap::kOk)
		{
			int jmax = pTab->CountRangeLines() ;

			const char *pTName ;
			pTab->GetName(pTName) ;
			acutPrintf ("\nRange Table %s:", pTName) ;

			// initialize AcMapRangeLine pointer to NULL before the loop
			AcMapRangeLine *pLine = NULL ;
			for (int j = 0; j < jmax; j++)
			{
				// pLine will be allocated at the first call
				if (pTab->GetRangeLine(pLine, j) == AcMap::kOk)
				{
					const char *pDef, *pRet;
					AcMap::ERangeOperator kOp ;
					pLine->GetRangeDefinition(pDef) ;
					pLine->GetRangeOperation(kOp) ;
					pLine->GetRangeValue(pRet) ;
					acutPrintf ("\n - %s %d %s", pDef, kOp, pRet) ;
				}
			}

			// don't forget to delete the object
			if(pLine)
				delete pLine ;
		}
	}

	// don't forget to delete the object
	if(pTab)
		delete pTab ;
}

/****************************************************************************/
/* Commands */
/****************************************************************************/
// implementation of EDITDSET command
void editDSet()
{
    AcMapSession	*mapApi ;
	AcMapProject    *pProj ;
    AcMapDrawingSet *pDSet ;
    char         res[32] ;   
   
    do {
		// get pointer to the root API object
        mapApi = AcMapGetSession();
        if (mapApi == NULL) 
        {
            acutPrintf ("\nCan't connect to MAP") ;
            break ;
        }
        
		// get the project, associated with the current AutoCAD document
		if (mapApi->GetProject(pProj) == Adesk::kFalse)
		{
		    acutPrintf ("\nCan't get MAP project") ;
            break ;
        }

		// get the drawing set, associated with the project
        if (pProj->GetDrawingSet(pDSet) == Adesk::kFalse)
        {
            acutPrintf ("\nCan't get drawing set") ;
            break ;
        }
        
        do {  
			// print the current drawing set state
            printDSet(pDSet) ;

			*res = EOS ;
            acedInitGet(0, "eXit aTtach Detach Activate deaCtivate Settings gettaBle aLiases View Preview Report Query Zoom") ; 
            if (acedGetKword ("\naTtach/Detach/Activate/deaCtivate/Settings/gettaBle/aLiases/View/Preview/Report/Query/Zoom/<eXit>: ", res) == RTNORM)
            {
                if (*res == EOS || !(strcmp(res, "eXit")))
                {
                    break ;
                }
                else if (!strcmp(res, "aTtach"))
                {
                    if (acedGetString (1, "Enter alias path: ", res) == RTNORM)
					{
						// initialize AcMapAttachedDrawing pointer to NULL
						AcMapAttachedDrawing *pDwg = NULL ;

						// attach an object to the drawing
						if (pDSet->AttachDrawing(pDwg, res) == AcMap::kOk)
						{
							// don't forget to delete the object
							delete pDwg ;
						}
					}
                }
				else if (!strcmp(res, "Detach"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// detach the drawing
					pDSet->DetachDrawing(res) ;
				}
                else if (!strcmp(res, "aLiases"))
				{
					// edit alias list
					editAliases() ;
				}
				else if (!strcmp(res, "Activate"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						// activate the drawing
						if(pDwg->Activate() != AcMap::kOk)
						{
							acutPrintf ("\nCan't activate drawing.") ;
						}

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "deaCtivate"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						// deactivate the drawing
						if(pDwg->Deactivate() != AcMap::kOk)
						{
							acutPrintf ("\nCan't deactivate drawing.") ;
						}

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "gettaBle"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						// get symbol tables - nested dialog
						getTable(pDwg) ;

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "Settings"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						// edit transformation settings - nested dialog
						editSettings(pDwg) ;

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "View"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						// preview the whole drawing
						if(pDwg->Preview() != AcMap::kOk)
						{
							acutPrintf ("\nCan't preview drawing.") ;
						}

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "Preview"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						AcDbObjectIdArray tIds ;

						// get Ids of the objects that match the current query
						if(pDwg->ApplyQuery(tIds) != AcMap::kOk)
						{
							acutPrintf ("\nCan't apply query to drawing.") ;
						}

						// draw these objects in preview mode
						if(pDwg->Preview(tIds) != AcMap::kOk)
						{
							acutPrintf ("\nCan't Preview.") ;
						}

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "Report"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						AcDbObjectIdArray tIds ;

						// get Ids of the objects that match the current query
						if(pDwg->ApplyQuery(tIds) != AcMap::kOk)
						{
							acutPrintf ("\nCan't apply query to drawing.") ;
						}

						// make report for these objects using the current template settings
						if(pDwg->Report(tIds) != AcMap::kOk)
						{
							acutPrintf ("\nCan't Report.") ;
						}

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "Query"))
				{
					acedGetString(1, "Enter alias path ", res) ;

					// initialize AcMapAttachedDrawing pointer to NULL 
					AcMapAttachedDrawing *pDwg = NULL ;
					if (pDSet->GetDrawing(pDwg, res, Adesk::kFalse) == AcMap::kOk)
					{
						AcDbObjectIdArray tIds ;

						// get Ids of the objects that match the current query
						if(pDwg->ApplyQuery(tIds) != AcMap::kOk)
						{
							acutPrintf ("\nCan't apply query to drawing.") ;
						}

						// clone objects from the source database to the project
						if(pDwg->QueryIn(tIds) != AcMap::kOk)
						{
							acutPrintf ("\nCan't query in") ;
						}

						// don't forget to delete the object
						delete pDwg ;
					}
				}
				else if (!strcmp(res, "Zoom"))
				{
					// zoom drawing extents to the combined extents of all 
					// Active or Locked attached drawings
					pDSet->ZoomExtents() ;
				}
			}
			else
            {
                break ;
            }

            printErrStack() ;

		} while(1) ;

	} while (0) ;
}

// implementation of EDITQUERY command
void
editQuery()
{
	AcMapQuery *pQuery = NULL ;
	char *pStr = NULL ;
	int code ;
	char res[125] ;

	do {
		if (pQuery && pQuery->GetStringContext(pStr) == AcMap::kOk)
		{
			acutPrintf ("\n%s", pStr) ;
			delete [] pStr ;
			Adesk::Boolean bIsDefined ;
			pQuery->IsPropertyAlterationDefined(bIsDefined) ;
			if (bIsDefined)
			{
				acutPrintf("\nProperty Alteration Defined") ;
			}
			pQuery->IsReportTemplateDefined(bIsDefined) ;
			if (bIsDefined)
			{
				acutPrintf("\nReport Template Defined") ;
			}
					
		}

		acedInitGet(0, "Alteration Define Execute Clear eXit Getcurrent setcuRrent rePort Test Save Load") ;
		code = acedGetKword("\nAlteration Define/Execute/Clear/Getcurrent/setcuRrent/rePort/Test/Save/Load/<eXit>: ", res) ;
		if (!stricmp(res, "eXit"))
		{
			break ;
		}
		else if (!stricmp(res, "Clear"))
		{
			if (pQuery)
			{
				pQuery->Clear() ;
			}
		}
		else if (!stricmp(res, "Alteration"))
		{
			if (pQuery)
			{
				AcMapPropertyAlterationDefinition *pDef ;
				if (pQuery->GetPropertyAlteration(pDef) == AcMap::kOk)
				{
					DefineAlteration(pDef) ;
					if (pDef->CountAlterations() > 0)
					{
						if(pQuery->EnablePropertyAlteration(Adesk::kTrue) != AcMap::kOk)
						{
							acutPrintf("\nCan't enable property alteration.") ;
						}
					}
				}
			}
		}
		else if (!stricmp(res, "rePort"))
		{
			if (pQuery)
			{
				AcMapReportTemplate *pTempl ;
				if (pQuery->GetReportTemplate(pTempl) == AcMap::kOk)
				{
					DefineTemplate(pTempl) ;
					if (pTempl->CountLines() > 0)
					{
						if(pQuery->SetMode(AcMap::kQueryReport) != AcMap::kOk)
						{
							acutPrintf("\nCan't set query mode to report.") ;
						}
					}
					delete pTempl ;
				}
			}
		}
		else if (!stricmp(res, "Define"))
		{
			AcMapQueryBranch tBranch ;
			if (define(tBranch))
			{
				if (!pQuery)
				{
					pQuery = getQuery (Adesk::kFalse) ;
				}

				if (pQuery)
				{
					if(pQuery->Define(&tBranch) != AcMap::kOk)
					{
						acutPrintf("\nCan't define query branch.") ;
					}
				}
			}
		}
		else if (!stricmp(res, "Execute"))
		{
			if (pQuery)
			{
				if(pQuery->SetMode(AcMap::kQueryDraw) != AcMap::kOk)
				{
					acutPrintf("\nCan't set query mode to draw.") ;
				}

				if(pQuery->Run() != AcMap::kOk)
				{
					acutPrintf("\nCan't run query.") ;
				}
			}
		}
		else if (!stricmp(res, "Getcurrent"))
		{
			if(pQuery)
			{
				delete pQuery ;
				pQuery = NULL ;
			}

			pQuery = getQuery(Adesk::kFalse) ;
			if (pQuery)
			{
				if(pQuery->LoadFromCurrent() != AcMap::kOk)
				{
					acutPrintf("\nCan't load query from current query.") ;
				}
			}
		}
		else if (!stricmp(res, "setcuRrent"))
		{
			if (pQuery == NULL)
			{
				pQuery = getQuery(Adesk::kTrue) ;
			}
			else
			{
				if(pQuery->SetCurrent() != AcMap::kOk)
				{
					acutPrintf("\nCan't set query to current query.") ;
				}
			}
		}
		else if (!stricmp(res, "Test"))
		{
			if (pQuery)
			{
				AcMapQueryBranch *pBranch = NULL ;
				if(pQuery->GetBranch(pBranch) == AcMap::kOk)
				{
					if(pQuery->Define(pBranch) != AcMap::kOk)
					{
						acutPrintf("\nCan't define query branch.") ;
					}
				}
			}
		}
		else if (!stricmp(res, "Save"))
		{
			if (pQuery)
			{
				saveQuery(pQuery) ;
			}
			else
			{
				acutPrintf ("\nNothing to save.") ;
			}
		}
		else if (!stricmp(res, "Load"))
		{
			if (pQuery == NULL)
			{
			 	pQuery = getQuery (Adesk::kFalse) ;
				
				if(pQuery)
				{
					load(pQuery) ;
				}
			}
		}
        printErrStack() ;

	} while (code == RTNORM) ;

	if(pQuery)
		delete pQuery ;
}

// implementation of TESTQLIB command
void testQueryLib()
{
	char buff[132] ;

	do {
		printQLib() ;
		acedInitGet (0, "Add Edit Remove eXit") ;
		if (acedGetKword ("\nAdd/Edit/Remove/<eXit>: ", buff) != RTNORM)
		{
			break ;
		}

		if (!strcmp(buff, "Add"))
		{
			addQuery() ;
		}
		else if (!strcmp(buff, "Edit"))
		{
			renameQuery() ;
		}
		else if (!strcmp(buff, "Remove"))
		{
			removeQuery() ;
		}
		else
		{
			break ;
		}
	} while (1) ;
}

// implementation of TESTOPTIONS command
void testOptions()
{
	AcMapSession *mapApi = AcMapGetSession();
	if (mapApi != NULL) 
	{
		AcMapProject *pProj ;
		char name[132] ;
		*name = EOS ;
		if (acedGetString(1, "\nEnter option name: ", name) == RTNORM
			&& *name != EOS)
		{
			int val ;
			char *pcVal ;

			if (mapApi->GetOptionValue(name, val) == AcMap::kOk)
			{
				CString buff ;
				int newVal ;
				buff.Format("\nEnter %s <%d>: ", name, val) ;
				if (acedGetInt (LPCTSTR(buff), &newVal) == RTNORM)
				{
					if(mapApi->SetOptionValue(name, newVal) != AcMap::kOk)
					{
						acutPrintf("\nCan't set option value.") ;
					}
				}
			}
			else if (mapApi->GetOptionValue(name, pcVal) == AcMap::kOk)
			{
				CString buff ;
				char newVal[128] ;
				buff.Format("\nEnter %s <%s>: ", name, pcVal) ;
				if (acedGetString (0, LPCTSTR(buff), newVal) == RTNORM &&
					*newVal != '\0')
				{
					if(mapApi->SetOptionValue(name, newVal) != AcMap::kOk)
					{
						acutPrintf("\nCan't set option value.") ;
					}
				}
				if(pcVal)
					delete [] pcVal ;
			}
			else if (mapApi->GetProject(pProj) == Adesk::kTrue)
			{
				if (pProj->GetOptionValue(name, val) == AcMap::kOk)
				{
					CString buff ;
					int newVal ;
					buff.Format("\nEnter %s <%d>: ", name, val) ;
					if (acedGetInt (LPCTSTR(buff), &newVal) == RTNORM)
					{
						if(mapApi->SetOptionValue(name, newVal) != AcMap::kOk)
						{
							acutPrintf("\nCan't set option value.") ;
						}
					}
				}
				else if (pProj->GetOptionValue(name, pcVal) == AcMap::kOk)
				{
					CString buff ;
					char newVal[128] ;
					buff.Format("\nEnter %s <%s>: ", name, pcVal) ;
					if (acedGetString (0, LPCTSTR(buff), newVal) == RTNORM
						&& *newVal != '\0')
					{
						if(mapApi->SetOptionValue(name, newVal) != AcMap::kOk)
						{
							acutPrintf("\nCan't set option value.") ;
						}
					}
					if(pcVal)
						delete [] pcVal ;
				}
				else
				{
					acutPrintf ("\nInvalid option name.") ;
				}
			}
			else
			{
				acutPrintf ("\nCan't access the current MAP project.") ;
			}
		}
	}
}

// implementation of TESTEXPR command
void testExpression()
{
	char			buff[132] ;
    AcMapSession	*mapApi = NULL ;
	AcMapProject	*pProj = NULL ;
	AcMapExpression *pExpress = NULL ;
	ads_point p ;
	ads_name ename;
	AcDbObjectId	Id = 0 ;
	AcMapValue		Val ;
   
    do {
		mapApi = AcMapGetSession();
        if (mapApi == NULL) 
        {
            acutPrintf ("\nCan't connect to MAP") ;
            break ;
        }
 
		AcMapProject *pProj ;
		if (mapApi->GetProject(pProj) == Adesk::kFalse)
		{
		    acutPrintf ("\nCan't get MAP project") ;
            break ;
        }
        
		*buff = EOS ;
		if (acedGetString (1, "Enter an expression: ", buff) != RTNORM ||
			*buff == EOS)
		{
			break ;
		}

		if (pProj->DefineExpression(pExpress, buff) != AcMap::kOk)
		{
			break ;
		}

		if (acedEntSel ("\nSelect object: ", ename, p) == RTNORM
			&& acdbGetObjectId(Id, ename) != Acad::eOk)
		{
			break ;
		}

		if (pExpress->Execute(Val, Id) == AcMap::kOk)
		{
			switch (Val.Type())
			{
				case AcMap::kInteger:
					acutPrintf ("\nResult = %d", (int)Val) ;
					break ;

				case AcMap::kReal:
					acutPrintf ("\nResult = %f", (double)Val) ;
					break ;

				case AcMap::kCharacter:
					acutPrintf ("\nResult = %s", (const char*)Val) ;
					break ;
					
				case AcMap::kPoint:
					{
						AcGePoint3d p = Val ;
						acutPrintf ("\nResult = %f,%f,%f", p[0],p[1],p[2]) ;
					}
					break ;
					
				default:
				acutPrintf ("\nUndefined result type.") ;
			}
		}
		else
		{
			acutPrintf ("\nExecution error.") ;
		}

	} while (0) ;

	delete pExpress ;
}

// implementation of TESTRANGE command
void testRange()
{
    AcMapRangeLibrary	*pLib = NULL ;
	char				buff[132] ;
	int					ret ;
   
	AcMapSession *mapApi = AcMapGetSession();
	if (mapApi != NULL) 
	{
		AcMapProject *pProj ;
		if (mapApi->GetProject(pProj) == Adesk::kTrue &&
			pProj->GetRangeLibrary(pLib))
		{
			do
			{
				printRangeLib(pLib) ;
				acedInitGet (0, "addTable Rename addLine Update eXit") ;
				ret = acedGetKword ("\naddTable/Rename/addLine/Update/<eXit>: ", buff) ;
				if (ret == RTNORM)
				{
					if (!stricmp(buff, "addTable"))
					{
						char pcName[132], pcDsc[132] ;
						*pcName = EOS ;
						*pcDsc = EOS ;
						if (acedGetString (1, "\nEnter table name: ", pcName) == RTNORM 
							&& acedGetString (1, "\nEnter table description: ", pcDsc) == RTNORM 
							&& *pcName != EOS)
						{
							AcMapRangeTable *pTable = NULL ;
							if (pLib->AddRangeTable(pTable, pcName, *pcDsc == EOS ? NULL : pcDsc) == AcMap::kOk)
							{
								acutPrintf ("\nDone.") ;
							}
							
							if(pTable)
								delete pTable ;
						}
					}
					else if (!stricmp(buff, "Rename"))
					{
						char pcOldName[132], pcNewName[132] ;
						*pcOldName = EOS ;
						*pcNewName = EOS ;
						if (acedGetString (1, "\nEnter old name: ", pcOldName) == RTNORM 
							&& acedGetString (1, "\nEnter new name: ", pcNewName) == RTNORM 
							&& *pcOldName != EOS && *pcNewName != EOS)
						{
							AcMapRangeTable *pTable = NULL ;
							if (pLib->GetRangeTable(pTable, pcOldName) == AcMap::kOk 
								&& pTable->SetName(pcNewName) == AcMap::kOk)
							{
								acutPrintf ("\nDone.") ;
							}
							
							if(pTable)
								delete pTable ;
						}
					}
					else if (!stricmp(buff, "addLine"))
					{
						char pcName[132];
						*pcName = EOS ;
						if (acedGetString (1, "\nEnter table name: ", pcName) == RTNORM 
							&& *pcName != EOS)
						{
							AcMapRangeTable *pTable = NULL ;
							if (pLib->GetRangeTable(pTable, pcName) == AcMap::kOk)
							{
								char pcDef[132] ;
								char pcRet[132] ;
								AcMap::ERangeOperator kOp ;
								if (GetRangeLineDef(kOp, pcDef, pcRet)
									&&pTable->AddRangeLine(kOp, *pcDef == EOS ? NULL : pcDef, pcRet) == AcMap::kOk)
								{
									acutPrintf ("\nDone.") ;
								}
							}
							
							if(pTable)
								delete pTable ;
						}
					}
					else if (!stricmp(buff, "Update"))
					{
						char pcName[132];
						*pcName = EOS ;
						if (acedGetString (1, "\nEnter table name: ", pcName) == RTNORM 
							&& *pcName != EOS)
						{
							AcMapRangeTable *pTable = NULL ;
							if (pLib->GetRangeTable(pTable, pcName) == AcMap::kOk)
							{
								int i ;
								if (acedGetInt ("\nEnter line number: ", &i) == RTNORM)
								{
									AcMapRangeLine *pLine = NULL ;
									if (pTable->GetRangeLine(pLine, i) == AcMap::kOk)
									{
										char pcDef[132] ;
										char pcRet[132] ;
										AcMap::ERangeOperator kOp ;
										if (GetRangeLineDef(kOp, pcDef, pcRet)
											&&pLine->SetRangeOperation(kOp) == AcMap::kOk
											&&pLine->SetRangeDefinition(*pcDef == EOS ? NULL : pcDef) == AcMap::kOk
											&&pLine->SetRangeValue(pcRet) == AcMap::kOk)
										{
											acutPrintf ("\nDone.") ;
										}
									}
									if(pLine)
										delete pLine ;
								}
							}
							if(pTable)
								delete pTable ;
						}	
					}
				}
			} while (ret == RTNORM) ;
		}
	}
}

// Main ARX entry point
AcRx::AppRetCode acrxEntryPoint(AcRx::AppMsgCode msg, void *appId)
{
   AcMapSession   *mapApi = NULL  ;
   
   switch (msg) {

   case AcRx::kInitAppMsg:
	   acrxRegisterAppMDIAware(appId);
       acedRegCmds->addCommand ("MAP_API_SAMPLE_GROUP",
                                "EDITALIASES",
                                "EDITALIASES",
                                ACRX_CMD_MODAL,
                                &editAliases);
       acedRegCmds->addCommand ("MAP_API_SAMPLE_GROUP",
                                "EDITDSET",
                                "EDITDSET",
                                ACRX_CMD_MODAL,
                                &editDSet);
       acedRegCmds->addCommand ("MAP_API_SAMPLE_GROUP",
                                "EDITQUERY",
                                "EDITQUERY",
                                ACRX_CMD_MODAL,
                                &editQuery);
       acedRegCmds->addCommand ("MAP_API_SAMPLE_GROUP",
                                "TESTQLIB",
                                "TESTQLIB",
                                ACRX_CMD_MODAL,
                                &testQueryLib);
       acedRegCmds->addCommand ("MAP_API_SAMPLE_GROUP",
                                "TESTOPTIONS",
                                "TESTOPTIONS",
                                ACRX_CMD_MODAL,
                                &testOptions);
       acedRegCmds->addCommand ("MAP_API_SAMPLE_GROUP",
                                "TESTEXPR",
                                "TESTEXPR",
                                ACRX_CMD_MODAL,
                                &testExpression);
       acedRegCmds->addCommand ("MAP_API_SAMPLE_GROUP",
                                "TESTRANGE",
                                "TESTRANGE",
                                ACRX_CMD_MODAL,
                                &testRange);

	   pDSetReact = new AcMyDrawingSetReactor() ;
	   pAliasReact = new AcMyAliasesReactor() ;
	   pQLibReact = new AcMyQueryLibraryReactor() ;
	   pPrefReact = new AcMyProjectOptionsReactor() ;
	   pSysOptReactor = new AcMySessionOptionsReactor () ;
       mapApi = AcMapGetSession();
	   if (mapApi)
	   {
			mapApi->AddOptionsReactor(pSysOptReactor) ;

			AcMapProject *pProj ;
			if (mapApi->GetProject(pProj))
			{
				AcMapDrawingSet *pDSet = NULL ;
				if (pProj->GetDrawingSet(pDSet))
				{
					pDSet->AddReactor(pDSetReact) ;
				}

				AcMapAliases *pAlias = NULL ;
				if (mapApi->GetAliases(pAlias))
				{
					pAlias->AddReactor(pAliasReact) ;
				}

				AcMapQueryLibrary *pQLib = NULL ;
				if (pProj->GetQueryLibrary(pQLib))
				{
					pQLib->AddReactor(pQLibReact) ;
				}
			
				pProj->AddOptionsReactor(pPrefReact) ;
		   }
	   }

	   acrxDynamicLinker->unlockApplication(appId);
	   break ;

   case AcRx::kUnloadAppMsg:
       acedRegCmds->removeGroup ("TOP_TEST_GROUP");
	   delete pDSetReact ;
	   pDSetReact = NULL ;
	   delete pAliasReact ;
	   pAliasReact = NULL ;
	   delete pQLibReact;
       pQLibReact = NULL;
       delete pPrefReact;
	   pPrefReact = NULL;
       delete pSysOptReactor;
       pSysOptReactor = NULL;
	   break ;

   case AcRx::kUnloadDwgMsg:
       mapApi = AcMapGetSession();
	   if (mapApi)
	   {
			AcMapProject *pProj ;
			if (mapApi->GetProject(pProj))
			{
				mapApi->RemoveOptionsReactor(pSysOptReactor) ;

				AcMapDrawingSet *pDSet = NULL ;
				if (pProj->GetDrawingSet(pDSet))
				{
					pDSet->RemoveReactor(pDSetReact) ;
				}

				AcMapAliases *pAlias = NULL ;
				if (mapApi->GetAliases(pAlias))
				{
					pAlias->RemoveReactor(pAliasReact) ;
				}

				AcMapQueryLibrary *pQLib = NULL ;
				if (pProj->GetQueryLibrary(pQLib))
				{
					pQLib->RemoveReactor(pQLibReact) ;
				}
			
				pProj->RemoveOptionsReactor(pPrefReact) ;
			}
		}  
		break ;
	}

	return AcRx::kRetOK;
}
/*EOF*/
